(function($){
  const $root = $('.kmd-reels-root');
  if(!$root.length) return;

  const settings = (window.KMD_REELS_DATA && KMD_REELS_DATA.settings) || {};
  const accent = settings.accent_color || '#ff7a00';
  document.documentElement.style.setProperty('--kmd-accent', accent);

  const $reels = $root.find('.kmd-reel');
  let index = (typeof window.KMD_REEL_ACTIVE_INDEX !== 'undefined') ? window.KMD_REEL_ACTIVE_INDEX : 0;

  function $cur(){ return $reels.eq(index); }

  // Background (desktop blur)
  function updateBackground(){
    const bg = $cur().data('bg') || '';
    if($root.data('blur') == 1 || $root.data('blur') === '1'){
      $root.find('.kmd-reels-overlay').css('background-image', bg ? 'url('+bg+')' : 'none');
    }
  }

  // Close -> back to referrer/home
  $root.on('click', '.kmd-close', function(){
    if(document.referrer){ window.history.back(); }
    else { window.location.href = (window.KMD_REELS_DATA && KMD_REELS_DATA.siteurl) || '/'; }
  });

  // Helpers
  function playVideo(v, withSound){
    if(!v) return;
    v.muted = !withSound;
    v.play().catch(()=>{});
  }

  // TikTok-like transition + single focus
  function setActive(i){
    index = (i + $reels.length) % $reels.length;
    $reels.removeClass('is-active');
    $cur().addClass('is-active');

    // Pause all, play current
    $reels.find('video').each(function(){ try{ this.pause(); }catch(e){} });
    const v = $cur().find('video').get(0);
    if(v){
      if(settings.autoplay_sound){
        playVideo(v, true);
        // If blocked, fallback to muted
        setTimeout(()=>{ if(v.paused) playVideo(v, false); }, 200);
      } else {
        playVideo(v, false);
      }
      bindProgress(v);
      bindCenterPlay(v);
    }

    refreshDownload();
    updateBackground();
    maybeShowAd();
  }

  // Show play button only when paused
  function bindCenterPlay(video){
    const $btn = $cur().find('.kmd-center-play');
    if(!$btn.length) return;
    function sync(){ if(video.paused) $btn.removeAttr('hidden'); else $btn.attr('hidden', true); }
    sync();
    video.addEventListener('play', sync);
    video.addEventListener('pause', sync);
    $btn.off('click').on('click', function(e){ e.preventDefault(); if(video.paused) video.play().catch(()=>{}); else video.pause(); });
  }

  // Progress -> glow top + bottom bar
  function bindProgress(video){
    const bar = $cur().find('.kmd-progress span').get(0);
    function tick(){
      if(video && video.duration){
        const p = (video.currentTime / video.duration) * 100;
        if(bar){ bar.style.width = p + '%'; }
        $cur().find('.kmd-frame').css('--kmd-progress', p + '%');
      }
      requestAnimationFrame(tick);
    }
    requestAnimationFrame(tick);
    video.onended = ()=> setActive(index+1);
  }

  // Controls
  $root.on('click','.kmd-mute', function(e){
    e.preventDefault();
    const v = $cur().find('video').get(0); if(!v) return;
    v.muted = !v.muted;
  });

  $root.on('click','.kmd-share', async function(e){
    e.preventDefault();
    const src = $cur().find('source').attr('src');
    if(navigator.share){ try{ await navigator.share({ url: src }); } catch(e){} }
    else { if(navigator.clipboard){ navigator.clipboard.writeText(src); alert('Link copied'); } }
  });

  // Download link -> set to current video's src
  function refreshDownload(){
    const src = $cur().find('source').attr('src') || '';
    $cur().find('.kmd-download').attr('href', src || '#');
  }

  // Navigation (wheel/touch)
  let touchStartY = 0;
  $root.on('touchstart', function(e){ touchStartY = e.originalEvent.touches[0].clientY; });
  $root.on('touchend', function(e){
    const y = e.originalEvent.changedTouches[0].clientY;
    if(Math.abs(y - touchStartY) > 40){ if(y < touchStartY) setActive(index+1); else setActive(index-1); }
  });
  $root.on('wheel', function(e){
    e.preventDefault();
    if(e.originalEvent.deltaY > 0) setActive(index+1); else setActive(index-1);
  }, { passive:false });

  // Desktop nav buttons
  $root.on('click','.kmd-nav-up', function(e){ e.preventDefault(); setActive(index-1); });
  $root.on('click','.kmd-nav-down', function(e){ e.preventDefault(); setActive(index+1); });

  // VAST Ads (simplified): show every X reels
  let reelCounter = 0;
  async function maybeShowAd(){
    if(!settings.vast_enabled) return;
    reelCounter++;
    const freq = parseInt(settings.ad_frequency || 3, 10);
    if(freq <= 0 || (reelCounter % freq !== 0)) return;

    const $frame = $cur().find('.kmd-frame');
    const $badge = $cur().find('.kmd-ad-badge');
    const $skip = $cur().find('.kmd-ad-skip');
    const skipAfter = parseInt(settings.ad_skip_seconds || 5, 10);

    const v = $cur().find('video').get(0);
    if(v){ try{ v.pause(); }catch(e){} }

    let $ad = $('<video class="kmd-video kmd-ad" preload="metadata" playsinline webkit-playsinline muted></video>');
    $frame.append($ad);
    $badge.removeAttr('hidden'); $skip.attr('hidden', true);

    async function parseVAST(url){
      try{
        const res = await fetch(url, { credentials:'omit' });
        const txt = await res.text();
        const m = txt.match(/<MediaFile[^>]*type=["\']video\/(mp4|webm)["\'][^>]*>([^<]+)<\/MediaFile>/i);
        if(m && m[2]) return m[2].trim();
      }catch(e){}
      return null;
    }
    let adSrc = null;
    if(settings.vast_tag_url){ adSrc = await parseVAST(settings.vast_tag_url); }
    if(!adSrc && settings.ad_fallback_url){ adSrc = settings.ad_fallback_url; }
    if(!adSrc){
      $badge.attr('hidden', true); $ad.remove();
      if(v){ v.play().catch(()=>{}); }
      return;
    }

    $ad.append('<source src="'+adSrc+'">');
    $ad.prop('muted', true);
    $ad.get(0).play().catch(()=>{});

    let timer = setTimeout(()=>{ $skip.removeAttr('hidden'); }, skipAfter * 1000);
    $skip.off('click').on('click', function(){
      try{ $ad.get(0).pause(); }catch(e){}
      $ad.remove(); $badge.attr('hidden', true);
      if(v){ v.play().catch(()=>{}); }
    });
    $ad.on('ended', function(){
      $ad.remove(); $badge.attr('hidden', true);
      if(v){ v.play().catch(()=>{}); }
    });
  }

  // Init
  setActive(index);

})(jQuery);
